# GLSL Shaders for TroikaTronix Isadora -- AI Reference Guide

## Purpose of This Document

This document is a complete technical reference for writing GLSL (OpenGL Shader Language) code that runs inside TroikaTronix Isadora's **GLSL Shader actor**. It is written so that an AI language model can generate correct, working shader code when a user asks for Isadora GLSL effects.

When generating code for Isadora, follow ALL rules in this document. Isadora uses an **older GLSL dialect** (compatible with GLSL 1.20 / OpenGL 2.1) and has its own **special comment syntax** for declaring parameters. Code that works on ShaderToy or in modern GLSL will often NOT compile in Isadora without adaptation.

---

## 1. The GLSL Shader Actor

The GLSL Shader actor is a built-in Isadora actor that compiles and runs a fragment shader (and optionally a vertex shader) on the GPU. The user double-clicks the actor to open a code editor, pastes in code, and clicks OK to compile.

### Actor Inputs and Outputs

| Port | Description |
|------|-------------|
| `video in 1` | Video stream -> available as `tex0` (sampler2D) |
| `video in 2` | Video stream -> available as `tex1` (sampler2D) |
| `video in 3` | Video stream -> available as `tex2` (sampler2D) |
| `video in 4` | Video stream -> available as `tex3` (sampler2D) |
| `Width` | Override output width (0 = auto from input or default resolution) |
| `Height` | Override output height (0 = auto from input or default resolution) |
| `Bypass` | When "on", passes input directly to output without processing |
| `video out` | The processed video output stream |
| `Trigger` | Sends a trigger whenever the output changes |

> **CRITICAL -- Ports only appear if uniforms are explicitly declared.** Video in ports (tex0, tex1 etc.) and parameter ports only appear on the actor if the corresponding `uniform` is explicitly declared in the shader code. Uniforms are NOT auto-injected in a way that creates ports. Always declare every uniform you use. See Section 3.

### Resolution Rules

- If both Width and Height are 0: output resolution matches the incoming video input. If there is no video input (generative shader), resolution falls back to the Default Resolution in Isadora Preferences.
- If both Width and Height are non-zero: output resolution is forced to those values.
- The actual resolution is available via the explicitly declared `iResolution` uniform.

---

## 2. GLSL Version and Dialect

### Critical: Isadora Uses an Old GLSL Version

Isadora's GLSL compiler targets **GLSL 1.20** (OpenGL 2.1 era). This means:

- Do **NOT** use `#version 130`, `#version 150`, `#version 330`, or any version directive above 120.
- Do **NOT** include any `#version` directive at all -- Isadora handles this internally.
- The keyword `texture` (modern GLSL) is **NOT available**. Use `texture2D()` instead.
- `in` and `out` qualifiers for varyings are **NOT available**. Use `varying` instead.
- `layout()` qualifiers are **NOT available**.
- Fragment shader output must be written to `gl_FragColor`, not to a user-defined `out vec4`.

### What Works

| Feature | Status |
|---------|--------|
| `texture2D(sampler, uv)` | YES -- Use this for all texture sampling |
| `gl_FragColor` | YES -- Use this for fragment output |
| `varying` | YES -- Use for vertex-to-fragment data |
| `uniform` | YES -- Use for all parameters and textures |
| Standard math functions (`sin`, `cos`, `mix`, `clamp`, `smoothstep`, `pow`, `abs`, `mod`, `floor`, `ceil`, `fract`, `dot`, `normalize`, `length`, `distance`, `reflect`, `refract`, `cross`, `min`, `max`, `step`, `atan`) | YES -- All available |
| `texture` (no "2D") | NO -- Does NOT compile |
| `#version 130` or higher | NO -- Causes errors |
| `in`/`out` varyings | NO -- Use `varying` |
| `layout(location=0) out vec4` | NO -- Use `gl_FragColor` |

---

## 3. Uniform Declarations -- CRITICAL RULES

### You MUST explicitly declare every uniform you use

This is the most important rule that differs from what you might expect. Even though Isadora provides built-in uniforms like `iTime`, `iResolution`, and `tex0`, **they will NOT create ports on the actor unless explicitly declared in your shader code.**

Always declare all uniforms you use in a single block, after all `ISADORA_FLOAT_PARAM` comments:

```glsl
// (all ISADORA_FLOAT_PARAM comments first -- see Section 4)

uniform float my_param;       // custom parameter
uniform float iTime;          // time in seconds -- MUST declare to use
uniform vec3 iResolution;     // resolution -- MUST declare to use
uniform sampler2D tex0;       // video in 1 -- MUST declare to create the port
uniform sampler2D tex1;       // video in 2 -- MUST declare to create the port
```

### Available built-in uniforms (must all be explicitly declared)

```glsl
uniform sampler2D tex0;              // Video input 1
uniform sampler2D tex1;              // Video input 2
uniform sampler2D tex2;              // Video input 3
uniform sampler2D tex3;              // Video input 4
uniform vec3      iResolution;       // xy = resolution in pixels, z = 1.0
uniform float     iTime;             // Current time in seconds
uniform float     iGlobalTime;       // Same as iTime
uniform vec4      iMouse;            // xy = mouse coords, zw = click coords
uniform vec4      iDate;             // x=year, y=month, z=day, w=seconds
uniform int       iFrame;            // Frame count since start
uniform sampler2D iChannel0;         // Same as tex0
uniform sampler2D iChannel1;         // Same as tex1
uniform sampler2D iChannel2;         // Same as tex2
uniform sampler2D iChannel3;         // Same as tex3
```

### Notes

- `tex0` and `iChannel0` refer to the same texture -- use either one, but declare whichever you use.
- `iResolution` is a `vec3` (z always 1.0). Use `iResolution.xy` for 2D resolution.
- Only declare `tex1`, `tex2`, `tex3` if your shader actually uses them -- each declaration creates a video input port.
- Declare `uniform int noise_mode;` etc. for integer uniforms tied to `ISADORA_INT_PARAM`.

---

## 4. Custom Parameters (Isadora Special Comment Syntax)

You create controllable input ports on the actor by writing specially formatted comments. **The structure is critical:** all parameter comments must be grouped together at the top, followed by all uniform declarations in a separate block. Do NOT interleave them.

### Correct Structure -- Parameters Then Uniforms

```glsl
// ISADORA_PLUGIN_DESC("Description of the shader.")

// ISADORA_FLOAT_PARAM(param_one, pon1, 0.0, 1.0, 0.5, "First parameter.")
// ISADORA_FLOAT_PARAM(param_two, ptw2, 0.0, 10.0, 1.0, "Second parameter.")
// ISADORA_INT_PARAM(mode, mode, 0, 3, 0, "Mode selector.")

uniform float param_one;
uniform float param_two;
uniform int   mode;
uniform float iTime;
uniform vec3  iResolution;
uniform sampler2D tex0;
```

### WRONG -- Do NOT interleave comments and uniforms

```glsl
// ISADORA_FLOAT_PARAM(param_one, pon1, 0.0, 1.0, 0.5, "First parameter.")
uniform float param_one;
// ISADORA_FLOAT_PARAM(param_two, ptw2, 0.0, 10.0, 1.0, "Second parameter.")
uniform float param_two;
```

The interleaved style may seem logical but does not reliably create ports in Isadora. Always use the grouped structure.

### Supported Parameter Types

#### Float Parameter
```glsl
// ISADORA_FLOAT_PARAM(brightness, brit, 0.0, 2.0, 1.0, "Brightness multiplier.")
```
Uniform type: `uniform float brightness;`

#### Integer Parameter
```glsl
// ISADORA_INT_PARAM(noise_mode, mode, 0, 4, 0, "Mode: 0=Classic 1=Turbulence 2=Ridged 3=Billowy 4=Warped.")
```
Uniform type: `uniform int noise_mode;`

Note: `ISADORA_INT_PARAM` uses integer values without decimals (e.g. `0, 4, 0`), not `0.0, 4.0, 0.0`.

#### Boolean Parameter
```glsl
// ISADORA_BOOL_PARAM(invert, invt, false, "Invert the image.")
```
Uniform type: `uniform bool invert;`

#### No Color Parameter

There is no `ISADORA_COLOR_PARAM`. To let users pick colors for a GLSL shader, use separate R/G/B float parameters. Two approaches:

- **0.0--1.0 range**: Use directly in the shader. Users must type values manually or use Calculator actors to scale.
- **0--255 range**: Compatible with Isadora's Color to RGBA and Color Maker RGBA actors, which output 0--255. Divide by 255.0 in the shader code. This lets users wire a Color control panel through Color to RGBA into the shader inputs.

### Syntax Rules (CRITICAL)

1. All `ISADORA_FLOAT_PARAM` / `ISADORA_INT_PARAM` comments must be **grouped together** before any `uniform` declarations.
2. The `variable_name` in the comment MUST exactly match the `uniform` variable name.
3. The `four_char_id` is a **unique identifier, ideally 4 characters** (letters and digits allowed, e.g. `brit`, `amnt`, `colr`, `uxy_`). Isadora uses this ID internally to maintain connections even if you reorder parameters. Shorter IDs (e.g. 3 characters) may work but 4 characters is the standard convention.
4. Every parameter MUST have a unique `four_char_id`. Reusing an ID causes conflicts.
5. Min, max, and default values use decimal notation for `ISADORA_FLOAT_PARAM` (e.g., `0.0`, not just `0`). Use integers without decimals for `ISADORA_INT_PARAM`.
6. The help text is shown when the user hovers over the input port.
7. There must be **NO space** between `ISADORA_FLOAT_PARAM` and the opening parenthesis.
8. Underscores in `variable_name` are displayed as spaces in the Isadora UI (e.g., `blur_amount` shows as "blur amount").

### Plugin Description

```glsl
// ISADORA_PLUGIN_DESC("This shader applies a custom effect.")
```

Place this at the very top of your shader, before any parameter comments.

---

## 5. Fragment Shader Structure

### Standard Template -- Use This as the Starting Point

This is the correct structure based on verified working shaders. Always follow this pattern:

```glsl
// =============================================
// Shader Name
// Made by [Author] [Year]
// =============================================
// Description of what this shader does.
// Connect video source to video in 1.
// =============================================

// ISADORA_PLUGIN_DESC("Short description of the shader.")

// ISADORA_FLOAT_PARAM(my_param, mypr, 0.0, 1.0, 0.5, "Parameter description.")

uniform float my_param;
uniform float iTime;
uniform vec3  iResolution;
uniform sampler2D tex0;

void main()
{
    vec2 uv = gl_FragCoord.xy / iResolution.xy;
    vec4 color = texture2D(tex0, uv);

    // === Your effect here ===
    vec3 result = color.rgb;

    gl_FragColor = vec4(result, color.a);
}
```

### Passthrough Shader (for feedback delay actors)

```glsl
// ISADORA_PLUGIN_DESC("Simple passthrough. Use as feedback delay in multi-actor chains.")

uniform vec3 iResolution;
uniform sampler2D tex0;

void main()
{
    vec2 uv = gl_FragCoord.xy / iResolution.xy;
    gl_FragColor = texture2D(tex0, uv);
}
```

### Generative Shader (no video input)

```glsl
// ISADORA_PLUGIN_DESC("Generative pattern. No video input needed.")

// ISADORA_FLOAT_PARAM(speed, sped, 0.0, 5.0, 1.0, "Animation speed.")
// ISADORA_FLOAT_PARAM(scale, scle, 0.1, 20.0, 4.0, "Pattern scale.")

uniform float speed;
uniform float scale;
uniform float iTime;
uniform vec3  iResolution;

void main()
{
    vec2 uv = gl_FragCoord.xy / iResolution.xy;
    float t = iTime * speed;

    float v = sin(uv.x * scale + t) * cos(uv.y * scale + t * 0.7);
    v = v * 0.5 + 0.5;

    gl_FragColor = vec4(vec3(v), 1.0);
}
```

### Important: Preserving Alpha

Always preserve the alpha channel unless you have a specific reason not to:

```glsl
// BAD -- destroys alpha, breaks Isadora compositing
gl_FragColor = vec4(result.rgb, 1.0);

// GOOD -- preserves original alpha
vec4 original = texture2D(tex0, uv);
gl_FragColor = vec4(result.rgb, original.a);
```

---

## 6. Coordinate Systems

### Fragment Coordinates

- `gl_FragCoord.xy` gives pixel coordinates (0,0 at bottom-left).
- To get normalized UV coordinates (0.0 to 1.0): `vec2 uv = gl_FragCoord.xy / iResolution.xy;`
- UV (0,0) is bottom-left, (1,1) is top-right.
- Isadora video textures have (0,0) at bottom-left by default.

### Pixel Size (Texel Size)

For effects that sample neighboring pixels:

```glsl
vec2 texel = 1.0 / iResolution.xy;
```

---

## 7. Porting ShaderToy Shaders to Isadora

### Step-by-Step Conversion

1. **Remove** any `#version` directive.
2. **Replace** `texture(` with `texture2D(`.
3. **Add explicit uniform declarations** for everything used: `iTime`, `iResolution`, `iChannel0` etc.
4. If using `mainImage(out vec4 fragColor, in vec2 fragCoord)` -- this works directly in Isadora.
5. If the shader references `iChannel0` etc., connect video sources to the corresponding `video in` ports.
6. **Add `ISADORA_FLOAT_PARAM` comments** (grouped at top) for any `uniform float` variables you want to control from Isadora's patching interface.
7. ShaderToy shaders using Buffer A / Buffer B (multi-pass) cannot be replicated in a single GLSL actor -- see Section 9.

### Common ShaderToy Replacements

| ShaderToy | Isadora Equivalent |
|-----------|-------------------|
| `texture(iChannel0, uv)` | `texture2D(tex0, uv)` |
| `fragColor = ...` (in mainImage) | Works as-is |
| `gl_FragColor = ...` (in main) | Works as-is |
| `iResolution.xy` | Works as-is -- but must be explicitly declared |
| `iTime` | Works as-is -- but must be explicitly declared |
| `iMouse` | Works as-is -- but must be explicitly declared |
| Buffer A / multi-pass | Chain multiple GLSL Shader actors (see Section 9) |

---

## 8. Vertex Shader

Most Isadora GLSL effects only need a fragment shader. The default vertex shader is:

```glsl
void main()
{
    gl_Position = gl_ModelViewProjectionMatrix * gl_Vertex;
    gl_TexCoord[0] = gl_MultiTexCoord0;
    gl_FrontColor = gl_Color;
}
```

In most cases, leave the vertex shader as default.

---

## 9. Multi-Pass Shaders and Feedback Loops

### The Problem

Isadora's GLSL Shader actor does NOT have a built-in persistent buffer. A single shader cannot access its own output from the previous frame. This is a critical limitation for effects like motion blur, trails, reaction-diffusion, and other temporal effects.

### The Solution: Chain Multiple GLSL Actors

To create feedback effects in Isadora:

1. **Sequential passes**: Connect the `video out` of one GLSL Shader actor to the `video in` of the next.
2. **Feedback loop**: Place a second passthrough GLSL actor between the output and the feedback input to create a one-frame delay.

### Pattern A: Video Input + Feedback

The most common feedback pattern. Actor A receives live video on tex0 and the previous frame (via Actor B) on tex1.

```
[Video Source]  -> [Actor A: video in 1]
[Actor A: out]  -> [Actor B: video in 1]
[Actor B: out]  -> [Actor A: video in 2]   <- feedback loop
[Actor A: out]  -> [Projector]
```

### Pattern B: Generative Feedback (No Video Input)

For self-generating effects that create their own content and feed it back. Actor A has no external video input -- it generates patterns internally and only receives its own previous frame via Actor B on tex0.

```
[Actor A: out] -> [Actor B: video in 1]
[Actor B: out] -> [Actor A: video in 1]  <- feedback only, no external video
[Actor A: out] -> [Projector]
```

In this pattern, the shader generates new content each frame (e.g. geometric rings, noise) and blends it with the warped/decayed feedback from tex0. Set Width and Height explicitly on both actors since there is no video input to determine resolution.

### Example: Feedback Trails (Pattern A)

**Actor A** (effect -- tex0 = current video, tex1 = previous frame from Actor B):

```glsl
// ISADORA_PLUGIN_DESC("Feedback trails effect. Connect video to in 1, Actor B to in 2.")

// ISADORA_FLOAT_PARAM(trail_amount, trla, 0.0, 0.99, 0.85, "Trail persistence. 0 = none, 0.99 = maximum.")

uniform float trail_amount;
uniform vec3  iResolution;
uniform sampler2D tex0;
uniform sampler2D tex1;

void main()
{
    vec2 uv = gl_FragCoord.xy / iResolution.xy;
    vec4 current  = texture2D(tex0, uv);
    vec4 previous = texture2D(tex1, uv);
    vec3 blended  = mix(current.rgb, previous.rgb, trail_amount);
    gl_FragColor  = vec4(blended, current.a);
}
```

**Actor B** (passthrough delay):

```glsl
// ISADORA_PLUGIN_DESC("Passthrough delay for feedback loop. Connect Actor A out to in 1, this out to Actor A in 2.")

uniform vec3 iResolution;
uniform sampler2D tex0;

void main()
{
    vec2 uv = gl_FragCoord.xy / iResolution.xy;
    gl_FragColor = texture2D(tex0, uv);
}
```

---

## 10. Common Utility Functions

Isadora's GLSL 1.20 has no `#include` mechanism, so utility functions must be inlined in every shader that uses them. The following are well-tested implementations used across many production shaders.

### Perceptual Luminance

Use the ITU-R BT.601 weights for perceptually correct luminance. Use this consistently across all shaders for predictable results:

```glsl
float luma = dot(color.rgb, vec3(0.299, 0.587, 0.114));
```

### RGB to HSV / HSV to RGB

```glsl
vec3 rgb2hsv(vec3 c)
{
    float cmax  = max(c.r, max(c.g, c.b));
    float cmin  = min(c.r, min(c.g, c.b));
    float delta = cmax - cmin;
    float h = 0.0;
    if (delta > 0.0001)
    {
        if      (cmax == c.r) h = mod((c.g - c.b) / delta, 6.0);
        else if (cmax == c.g) h = (c.b - c.r) / delta + 2.0;
        else                  h = (c.r - c.g) / delta + 4.0;
        h /= 6.0;
        if (h < 0.0) h += 1.0;
    }
    float s = (cmax < 0.0001) ? 0.0 : delta / cmax;
    return vec3(h, s, cmax);
}

vec3 hsv2rgb(vec3 c)
{
    float h = c.x * 6.0;
    float s = c.y;
    float v = c.z;
    float i = floor(h);
    float f = h - i;
    float p = v * (1.0 - s);
    float q = v * (1.0 - s * f);
    float t = v * (1.0 - s * (1.0 - f));
    if      (i < 1.0) return vec3(v, t, p);
    else if (i < 2.0) return vec3(q, v, p);
    else if (i < 3.0) return vec3(p, v, t);
    else if (i < 4.0) return vec3(p, q, v);
    else if (i < 5.0) return vec3(t, p, v);
    else               return vec3(v, p, q);
}
```

### Pseudo-Random Hash (for noise/grain)

```glsl
float rand(vec2 co)
{
    return fract(sin(dot(co, vec2(12.9898, 78.233))) * 43758.5453);
}
```

---

## 11. Complete Shader Examples

Each example demonstrates a distinct shader architecture pattern. Together they cover the most common effect types you will need to build.

### Example 1: Neighbor Sampling -- Edge Detection (Sobel)

Demonstrates sampling surrounding pixels using texel offsets for convolution-based effects (blur, sharpen, edge detect, emboss).

```glsl
// =============================================
// FD Sobel Edge Detect
// =============================================
// Detects edges using a Sobel convolution kernel.
// Connect video source to video in 1.
// =============================================

// ISADORA_PLUGIN_DESC("Sobel edge detection with adjustable threshold.")

// ISADORA_FLOAT_PARAM(threshold, thrs, 0.0, 1.0, 0.1, "Edge detection threshold.")
// ISADORA_FLOAT_PARAM(edge_brightness, edbr, 0.0, 5.0, 1.0, "Brightness of detected edges.")

uniform float threshold;
uniform float edge_brightness;
uniform vec3  iResolution;
uniform sampler2D tex0;

void main()
{
    vec2 uv    = gl_FragCoord.xy / iResolution.xy;
    vec2 texel = 1.0 / iResolution.xy;

    // Sample 3x3 neighborhood, convert to luminance
    float tl = dot(texture2D(tex0, uv + vec2(-texel.x,  texel.y)).rgb, vec3(0.299, 0.587, 0.114));
    float tc = dot(texture2D(tex0, uv + vec2(      0.0,  texel.y)).rgb, vec3(0.299, 0.587, 0.114));
    float tr = dot(texture2D(tex0, uv + vec2( texel.x,  texel.y)).rgb, vec3(0.299, 0.587, 0.114));
    float ml = dot(texture2D(tex0, uv + vec2(-texel.x,       0.0)).rgb, vec3(0.299, 0.587, 0.114));
    float mr = dot(texture2D(tex0, uv + vec2( texel.x,       0.0)).rgb, vec3(0.299, 0.587, 0.114));
    float bl = dot(texture2D(tex0, uv + vec2(-texel.x, -texel.y)).rgb, vec3(0.299, 0.587, 0.114));
    float bc = dot(texture2D(tex0, uv + vec2(      0.0, -texel.y)).rgb, vec3(0.299, 0.587, 0.114));
    float br = dot(texture2D(tex0, uv + vec2( texel.x, -texel.y)).rgb, vec3(0.299, 0.587, 0.114));

    // Sobel kernels
    float gx   = -tl - 2.0*ml - bl + tr + 2.0*mr + br;
    float gy   = -tl - 2.0*tc - tr + bl + 2.0*bc + br;
    float edge = step(threshold, sqrt(gx*gx + gy*gy)) * edge_brightness;

    vec4 original = texture2D(tex0, uv);
    gl_FragColor  = vec4(vec3(edge), original.a);
}
```

### Example 2: Generative Pattern (No Video Input)

Demonstrates a shader that generates its own output with no video source connected. Uses `iTime` for animation. Note: set Width and Height on the actor explicitly, or output resolution falls back to Isadora's default.

```glsl
// =============================================
// FD Plasma Generator
// =============================================
// Animated plasma pattern. No video input needed.
// =============================================

// ISADORA_PLUGIN_DESC("Animated plasma pattern generator. No video input needed.")

// ISADORA_FLOAT_PARAM(speed, sped, 0.0, 5.0, 1.0, "Animation speed.")
// ISADORA_FLOAT_PARAM(scale, scle, 0.1, 20.0, 4.0, "Pattern scale.")

uniform float speed;
uniform float scale;
uniform float iTime;
uniform vec3  iResolution;

void main()
{
    vec2  uv = gl_FragCoord.xy / iResolution.xy;
    float t  = iTime * speed;

    float v1 = sin(uv.x * scale + t);
    float v2 = sin(uv.y * scale + t * 0.7);
    float v3 = sin((uv.x + uv.y) * scale * 0.5 + t * 1.3);
    float v4 = sin(length(uv - 0.5) * scale * 2.0 - t);
    float v  = (v1 + v2 + v3 + v4) * 0.25;

    vec3 color;
    color.r = sin(v * 3.14159) * 0.5 + 0.5;
    color.g = sin(v * 3.14159 + 2.094) * 0.5 + 0.5;
    color.b = sin(v * 3.14159 + 4.189) * 0.5 + 0.5;

    gl_FragColor = vec4(color, 1.0);
}
```

### Example 3: Multi-Sample Filtering -- Horizontal Chroma Blur

Demonstrates weighted multi-tap sampling for blur/filter effects. This pattern separates luma and chroma, blurs only the chrominance horizontally (simulating analog video bandwidth limits), then recombines with the sharp luminance. Shows asymmetric kernel weights and luma-preserving recombination.

```glsl
// =============================================
// FD Chroma Blur
// =============================================
// Blurs chroma horizontally while keeping
// luma sharp. Simulates analog video bandwidth.
// Connect video source to video in 1.
// =============================================

// ISADORA_PLUGIN_DESC("Horizontal chroma blur with sharp luma. Simulates analog video chroma bandwidth.")

// ISADORA_FLOAT_PARAM(chroma_blur_width, cbwt, 0.0, 1.0, 0.35, "How wide chroma spreads horizontally. 0 = no blur, 1 = heavy smear.")
// ISADORA_FLOAT_PARAM(luma_sharpness, lmsh, 0.0, 1.0, 1.0, "Sharpness of luma. 1 = luma fully unaffected, 0 = both equally blurred.")

uniform float chroma_blur_width;
uniform float luma_sharpness;
uniform vec3 iResolution;
uniform sampler2D tex0;

void main()
{
    vec2 uv = gl_FragCoord.xy / iResolution.xy;
    vec2 texel = 1.0 / iResolution.xy;

    // Original pixel for sharp luma reference
    vec4 orig = texture2D(tex0, uv);
    float luma = dot(orig.rgb, vec3(0.299, 0.587, 0.114));

    // Horizontal chroma blur: asymmetric low-pass filter (more smear right)
    float blurPixels = chroma_blur_width * 24.0;

    // Gaussian-like weights
    float w0 = 0.227;
    float w1 = 0.194;
    float w2 = 0.144;
    float w3 = 0.094;
    float w4 = 0.053;

    // Sample center + 4 taps right (full weight) + 4 taps left (reduced weight)
    vec3 c0  = texture2D(tex0, uv).rgb;
    vec3 cR1 = texture2D(tex0, uv + vec2(texel.x * blurPixels * 0.5,  0.0)).rgb;
    vec3 cR2 = texture2D(tex0, uv + vec2(texel.x * blurPixels * 1.0,  0.0)).rgb;
    vec3 cR3 = texture2D(tex0, uv + vec2(texel.x * blurPixels * 1.5,  0.0)).rgb;
    vec3 cR4 = texture2D(tex0, uv + vec2(texel.x * blurPixels * 2.0,  0.0)).rgb;
    vec3 cL1 = texture2D(tex0, uv - vec2(texel.x * blurPixels * 0.25, 0.0)).rgb;
    vec3 cL2 = texture2D(tex0, uv - vec2(texel.x * blurPixels * 0.5,  0.0)).rgb;
    vec3 cL3 = texture2D(tex0, uv - vec2(texel.x * blurPixels * 0.75, 0.0)).rgb;
    vec3 cL4 = texture2D(tex0, uv - vec2(texel.x * blurPixels * 1.0,  0.0)).rgb;

    vec3 chromaBlurred = c0  * w0
                       + cR1 * w1 + cR2 * w2 + cR3 * w3 + cR4 * w4
                       + cL1 * (w1 * 0.6) + cL2 * (w2 * 0.6)
                       + cL3 * (w3 * 0.6) + cL4 * (w4 * 0.6);

    float weightSum = w0 + w1 + w2 + w3 + w4
                    + (w1 + w2 + w3 + w4) * 0.6;
    chromaBlurred /= weightSum;

    // Recombine: sharp luma + blurred chrominance
    float chromaLuma = dot(chromaBlurred, vec3(0.299, 0.587, 0.114));
    vec3 chrominance = chromaBlurred - vec3(chromaLuma);
    float finalLuma = mix(chromaLuma, luma, luma_sharpness);

    vec3 result = clamp(vec3(finalLuma) + chrominance, 0.0, 1.0);
    gl_FragColor = vec4(result, orig.a);
}
```

### Example 4: Two-Input Displacement

Demonstrates using two video inputs (tex0 + tex1). Source video on tex0, displacement map on tex1. Reads brightness from the map to offset sampling of the source. Shows 1D/2D displacement modes and edge wrapping.

```glsl
// =============================================
// FD Displace
// =============================================
// Offsets source video pixels using a
// displacement map. Source -> video in 1,
// map -> video in 2.
// =============================================

// ISADORA_PLUGIN_DESC("Displace - displaces source using a map. Source to in 1, map to in 2.")

// ISADORA_FLOAT_PARAM(amount, amnt, 0.0, 1.0, 0.1, "Displacement strength.")
// ISADORA_FLOAT_PARAM(angle, angl, 0.0, 360.0, 0.0, "Direction in degrees (1D mode).")
// ISADORA_FLOAT_PARAM(use_xy, uxy_, 0.0, 1.0, 0.0, "2D mode when > 0.5: R drives X, G drives Y.")
// ISADORA_FLOAT_PARAM(do_wrap, wrap, 0.0, 1.0, 1.0, "Wrap pixels around edges when > 0.5.")

uniform float amount;
uniform float angle;
uniform float use_xy;
uniform float do_wrap;
uniform vec3 iResolution;
uniform sampler2D tex0;
uniform sampler2D tex1;

void main()
{
    vec2 uv = gl_FragCoord.xy / iResolution.xy;

    vec4 dispSample = texture2D(tex1, uv);
    vec3 dRGB = dispSample.rgb;
    float dLuma = dot(dRGB, vec3(0.299, 0.587, 0.114));

    // 1D mode: displace along angle using luminance
    float displacement = (dLuma - 0.5) * amount * 2.0;
    float rad = radians(angle);
    vec2 dispVec1D = vec2(cos(rad), sin(rad)) * displacement;

    // 2D mode: R drives X, G drives Y
    vec2 dispVec2D = vec2(
        (dRGB.r - 0.5) * amount * 2.0,
        (dRGB.g - 0.5) * amount * 2.0
    );

    // Blend between 1D and 2D based on use_xy
    float mode2d = step(0.5, use_xy);
    vec2 displaceVec = mix(dispVec1D, dispVec2D, mode2d);
    vec2 newUV = uv + displaceVec;

    // Wrap or clamp
    float wrapOn = step(0.5, do_wrap);
    vec2 sampledUV = mix(clamp(newUV, 0.0, 1.0), fract(newUV + 1.0), wrapOn);

    vec4 displaced = texture2D(tex0, sampledUV);
    gl_FragColor = displaced;
}
```

### Example 5: Domain Warp Feedback with Hue Rotation

Demonstrates the feedback loop pattern (Section 9, Pattern A) with a wave-field UV warp and hue rotation applied to the previous frame. Shows HSV color space conversion (Section 10), time-based wave fields, and blending live video with persistent feedback structures. Requires a passthrough Actor B for the feedback delay.

```glsl
// =============================================
// FD Domain Warp Feedback
// =============================================
// Warps previous frame with a wave field,
// rotates hue in the feedback loop, blends
// with live video.
//
// Patching:
//   [Video Source]  -> [Actor A: video in 1]
//   [Actor A: out]  -> [Actor B: video in 1]
//   [Actor B: out]  -> [Actor A: video in 2]  <- feedback
//   [Actor A: out]  -> [Projector]
// =============================================

// ISADORA_PLUGIN_DESC("Domain warp feedback - warps previous frame, rotates hue, blends with live video. Connect video to in 1, Actor B feedback to in 2.")

// ISADORA_FLOAT_PARAM(warp_strength, wstr, 0.0, 0.3, 0.06, "How strongly feedback is deformed each frame.")
// ISADORA_FLOAT_PARAM(warp_speed, wspd, 0.0, 3.0, 0.5, "Wave field animation speed.")
// ISADORA_FLOAT_PARAM(decay, deca, 0.0, 0.99, 0.88, "How long feedback persists. High = deep trails.")
// ISADORA_FLOAT_PARAM(feedback_mix, fbmx, 0.0, 1.0, 0.5, "0 = pure video, 1 = pure feedback.")
// ISADORA_FLOAT_PARAM(color_shift, clsh, 0.0, 1.0, 0.05, "Hue rotation speed in feedback loop.")

uniform float warp_strength;
uniform float warp_speed;
uniform float decay;
uniform float feedback_mix;
uniform float color_shift;
uniform float iTime;
uniform vec3  iResolution;
uniform sampler2D tex0;
uniform sampler2D tex1;

vec3 rgb2hsv(vec3 c)
{
    float cmax  = max(c.r, max(c.g, c.b));
    float cmin  = min(c.r, min(c.g, c.b));
    float delta = cmax - cmin;
    float h = 0.0;
    if (delta > 0.0001)
    {
        if      (cmax == c.r) h = mod((c.g - c.b) / delta, 6.0);
        else if (cmax == c.g) h = (c.b - c.r) / delta + 2.0;
        else                  h = (c.r - c.g) / delta + 4.0;
        h /= 6.0;
        if (h < 0.0) h += 1.0;
    }
    float s = (cmax < 0.0001) ? 0.0 : delta / cmax;
    return vec3(h, s, cmax);
}

vec3 hsv2rgb(vec3 c)
{
    float h = c.x * 6.0;
    float s = c.y;
    float v = c.z;
    float i = floor(h);
    float f = h - i;
    float p = v * (1.0 - s);
    float q = v * (1.0 - s * f);
    float t = v * (1.0 - s * (1.0 - f));
    if      (i < 1.0) return vec3(v, t, p);
    else if (i < 2.0) return vec3(q, v, p);
    else if (i < 3.0) return vec3(p, v, t);
    else if (i < 4.0) return vec3(p, q, v);
    else if (i < 5.0) return vec3(t, p, v);
    else               return vec3(v, p, q);
}

void main()
{
    vec2 uv = gl_FragCoord.xy / iResolution.xy;
    float t = iTime * warp_speed;

    // Two-layer wave field for organic motion
    float wx = sin(uv.y * 4.1 + t) * cos(uv.x * 3.7 - t * 0.8)
             + sin(uv.y * 7.3 - t * 0.6 + 1.2) * 0.5;
    float wy = cos(uv.x * 4.6 + t * 0.9) * sin(uv.y * 3.1 + t * 1.1)
             + cos(uv.x * 6.8 - t * 0.7 + 0.7) * 0.5;

    vec2 warpedUV = clamp(uv + vec2(wx, wy) * warp_strength * (1.0 / 1.5), 0.0, 1.0);

    // Sample previous frame with hue rotation
    vec4 prevFrame = texture2D(tex1, warpedUV);
    vec3 prevHSV   = rgb2hsv(prevFrame.rgb);
    prevHSV.x      = mod(prevHSV.x + iTime * color_shift * 0.5, 1.0);
    vec3 feedback  = hsv2rgb(prevHSV) * decay;

    // Blend with live video
    vec4 videoInput = texture2D(tex0, uv);
    vec3 result = clamp(
        mix(videoInput.rgb, feedback, feedback_mix)
        + videoInput.rgb * (1.0 - feedback_mix) * 0.3,
        0.0, 1.0
    );

    gl_FragColor = vec4(result, videoInput.a);
}
```

---

## 12. Saving Shaders as Reusable Plugins

GLSL shaders can be saved as `.txt` files that appear in Isadora's Toolbox under the "GLSL Shaders" category.

### Save Location

- **macOS**: `~/Library/Application Support/TroikaTronix/Isadora/GLSL Shaders/`
- **Windows**: `%APPDATA%\TroikaTronix\Isadora\GLSL Shaders\`

The file extension must be `.txt`. The filename (without extension) becomes the actor name in the Toolbox.

### Naming Convention

Use a consistent prefix and capitalized words for clarity in the Toolbox, e.g.:

```
FD Fluid UV Distort.txt
FD VHS Tape Degradation.txt
FD Domain Warp Feedback A.txt
```

---

## 13. Common Mistakes and Troubleshooting

### No ports appear on the actor

**Cause**: Uniforms not explicitly declared in the shader.
**Fix**: Declare every uniform you use -- `tex0`, `iTime`, `iResolution` and all custom parameters -- as explicit `uniform` declarations in the code.

### Parameter ports appear but don't respond

**Cause**: `ISADORA_FLOAT_PARAM` comments interleaved with `uniform` declarations instead of grouped at top.
**Fix**: Move all `ISADORA_FLOAT_PARAM` comments into a single block at the top, then put all `uniform` declarations together in a block below.

### Shader compiles but output is black

**Cause**: Generative shader (no video input) with Width and Height both set to 0. Resolution falls back to Isadora's default, which may not trigger rendering as expected.
**Fix**: Set Width and Height explicitly on the actor (e.g. 1920 and 1080).

### Compile Errors

| Error | Cause | Fix |
|-------|-------|-----|
| `'texture' : no matching overloaded function found` | Using modern `texture()` | Replace with `texture2D()` |
| `'<' : syntax error` in ISADORA_FLOAT_PARAM line | Space before parenthesis | Ensure `// ISADORA_FLOAT_PARAM(` has no space before `(` |
| `FRAG ERROR: ... syntax error` | `#version` directive present | Remove any `#version` line |
| Shader compiles but no video output | tex0 not declared or not connected | Add `uniform sampler2D tex0;` and connect video to `video in 1` |

### Performance Tips

- Keep `for` loop iteration counts reasonable. More than 64 samples per pixel can cause frame drops.
- Use `const int` for loop bounds -- some GPU drivers reject variable loop bounds in GLSL 1.20.
- Avoid `discard` in fragment shaders when possible.
- For generative shaders with no video input, set Width and Height explicitly on the actor.

### Alpha Channel Issues

- Isadora's compositing depends on correct alpha values.
- Always carry the original alpha through: `gl_FragColor = vec4(result.rgb, original.a);`
- When combining multiple texture samples (e.g. blur), use the alpha from the center/original sample only.

---

## 14. Limitations of Isadora's GLSL Shader Actor

- **No persistent buffer**: Cannot access its own output from the previous frame. Use feedback loops (Section 9).
- **No compute shaders**: Only vertex and fragment shaders supported.
- **No geometry shaders or tessellation**.
- **No SSBO, UBO, or image load/store**.
- **Maximum 4 texture inputs** (tex0--tex3).
- **GLSL 1.20 only**: No integer textures, no bitwise ops in older drivers.
- **No `const int` from uniform for array sizes**: Array sizes must be compile-time constants.
- **No color parameter type**: See Section 4 for workarounds using R/G/B float parameters.

---

## 15. Converting ISF (Interactive Shader Format) Shaders

ISF shaders from sites like isf.video use a JSON header for parameters. To convert:

1. Remove the entire JSON header (everything between `/*{` and `}*/`).
2. Replace ISF input declarations with Isadora's grouped comment syntax (all at top, uniforms below).
3. Replace `IMG_NORM_PIXEL(inputImage, uv)` with `texture2D(tex0, uv)`.
4. Replace `IMG_PIXEL(inputImage, pixelCoord)` with `texture2D(tex0, pixelCoord / iResolution.xy)`.
5. Replace `isf_FragNormCoord` with `gl_FragCoord.xy / iResolution.xy`.
6. Replace `RENDERSIZE` with `iResolution.xy`.
7. Replace `TIME` with `iTime`.
8. Add explicit `uniform` declarations for everything used.
9. ISF `point2D` type becomes two separate float parameters (x and y).

---

## 16. Summary of Critical Rules for AI Code Generation

1. **Never** use `texture()` -- always `texture2D()`.
2. **Never** include `#version` directives.
3. **Never** use `in`/`out` varyings -- use `varying`.
4. **Always** write to `gl_FragColor`.
5. **Always** preserve the alpha channel: `gl_FragColor = vec4(result.rgb, original.a);`
6. **Always** group all `ISADORA_FLOAT_PARAM` / `ISADORA_INT_PARAM` comments together at the top, then put ALL `uniform` declarations together in a single block below. Never interleave them.
7. **Always** explicitly declare every uniform used -- `tex0`, `tex1`, `iTime`, `iResolution` etc. They do NOT auto-create ports unless declared.
8. **Always** give each parameter a unique 4-character ID.
9. Use `iResolution.xy` for resolution (declared as `uniform vec3 iResolution`).
10. Use `tex0`--`tex3` for texture inputs -- declare only the ones you use.
11. For feedback effects, chain two GLSL actors with a passthrough delay -- a single actor cannot access its previous frame.
12. Follow the file naming convention: `FD Shader Name.txt` with spaces, capital words, `.txt` extension.
13. Use `vec3(0.299, 0.587, 0.114)` for perceptual luminance (BT.601) consistently.
14. Use the `rgb2hsv` / `hsv2rgb` implementations from Section 10 for HSV color space conversion.
